package com.example.android.vpntest;

import android.app.Notification;
import android.app.NotificationChannel;
import android.app.NotificationManager;
import android.content.Intent;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.net.VpnService;
import android.os.Binder;
import android.os.Build;
import android.os.IBinder;
import android.os.Parcel;
import android.os.ParcelFileDescriptor;
import android.os.RemoteException;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.v4.app.NotificationCompat;
import android.support.v4.app.NotificationManagerCompat;
import android.util.Log;

import org.pcap4j.packet.DnsDomainName;
import org.pcap4j.packet.DnsPacket;
import org.pcap4j.packet.DnsQuestion;
import org.pcap4j.packet.DnsResourceRecord;
import org.pcap4j.packet.IllegalRawDataException;
import org.pcap4j.packet.IpPacket;
import org.pcap4j.packet.IpSelector;
import org.pcap4j.packet.IpV4Packet;
import org.pcap4j.packet.TcpPacket;
import org.pcap4j.packet.UdpPacket;
import org.pcap4j.packet.namednumber.IpNumber;
import org.pcap4j.packet.namednumber.TcpPort;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.net.Inet4Address;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.UnknownHostException;
import java.nio.ByteBuffer;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.DatagramChannel;
import java.nio.channels.FileChannel;
import java.nio.channels.SelectionKey;
import java.nio.channels.Selector;
import java.nio.channels.SocketChannel;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.Set;

// VpnService implementation based on the Tutorial on: https://www.thegeekstuff.com/2014/06/Android-vpn-service/

/**
 * Implements an Android VPN service. All packages from and to the internet are handled to the file descriptors inside this class.
 * Is able to inspect and modify IP packages. Currently the Protocols HTTP, DNS and IP are parsed and modified. Uses a list to
 * filter specific domains and IPs.
 * Base for this class is the tutorial found on https://www.thegeekstuff.com/2014/06/Android-vpn-service/
 * @author Steven Schalhorn
 * @author https://www.thegeekstuff.com/2014/06/Android-vpn-service/
 * @version 1.0
 */
public class UserVpnService extends VpnService {
    private static final String TAG = "VPN";
    private static final String CHANNEL_ID = "tfk-filter";
    private static final int NOTIFICATION_ID = 1337;

    //
    //
    /**
     * enable outside checks whether the service is running
     * will be reset on unforseen process termination
     */
    public static Boolean serviceRunning = false;

    // declare selector to manage Connections
    private static Selector mSelector;

    // open source ports
    private static HashSet<Integer> mTcpChannels = new HashSet<>();

    // Count blocks per App
    private static HashMap<String, Integer> appBlockCount = new HashMap<String, Integer>();

    // helper vars for redirected dns requests
    private static InetAddress currRedirectIp;
    private static HashMap<InetAddress, ArrayList<DnsDomainName>> domainNames = new HashMap<InetAddress, ArrayList<DnsDomainName>>();

    // Builder to set up VPN service
    private Builder builder = new Builder();

    // binder to control the service from outside
    private final IBinder binder = new UserVpnBinder();
    private Thread mThread;
    private ParcelFileDescriptor mInterface;

    private FileInputStream in;
    private FileOutputStream out;

    private static ListFilter filter = null;
    private ArrayList<Integer> categories = null;


    /**
     * Build the interface for the VPN service and start it.
     *
     * {@inheritDoc}
     */
    @Override
    public int onStartCommand(Intent intent, int flags, int startId) {
        try {
            // Selector to manage Connections
            mSelector = Selector.open();
        } catch (IOException e) {
            e.printStackTrace();
        }

        categories = intent.getIntegerArrayListExtra("categories");
        filter = new ListFilter(getApplicationContext(), categories);

        try {
            currRedirectIp = Inet4Address.getByName(PkgUtils.REDIRECT_IP);
        } catch (UnknownHostException e) {
            e.printStackTrace();
        }

        mThread = new Thread(new Runnable() {
            @Override
            public void run() {
                try {
                    // Build and set up the interface
                    mInterface = builder.setSession("TIME for kids - Filterdienst")
                            .addAddress("10.0.0.10", 24)
                            .addRoute("0.0.0.0", 0)
                            .addDnsServer("8.8.8.8")
                            .establish();

                    // never cross the streams: https://media1.tenor.com/images/daf67e34c4ea266f85f61540da0ea954/tenor.gif
                    in = new FileInputStream(mInterface.getFileDescriptor());
                    out = new FileOutputStream(mInterface.getFileDescriptor());

                    while (true) {
                        // Initialize byte array with 65535 bytes = max size of an IP Packet
                        // be a little bit smaller to account for pcap4j overhead
                        byte[] ipPacketBuffer = new byte[60000];

                        // If data is available, read it and process it for the designated channel
                        int ipPacketLength;
                        try {
                            ipPacketLength = in.read(ipPacketBuffer);
                        } catch (NullPointerException e) {
                            ipPacketLength = 0;
                            Log.d("PassException", "Exception Handling. Packet from App was empty.");
                        }

                        if (ipPacketLength > 0) {
                            try {
                                handleAppToInet(ipPacketBuffer, ipPacketLength);
                            } catch (IllegalRawDataException e) {
                                // couldn't create package from buffer
                                e.printStackTrace();
                            }
                        }

                        try {
                            handleInetToApp();
                        } catch (IOException e) {
                            e.printStackTrace();
                        }

                        terminateChannels();

                        Thread.sleep(3);
                    }

                } catch (Exception e) {
                    // Catch any exception
                    e.printStackTrace();
                } finally {
                    try {
                        if (mInterface != null) {
                            mInterface.close();
                            mInterface = null;
                        }
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }
            }

        }, "VPNServiceThread");

        //start the service
        mThread.start();

        Notification notification = createNotification();
        startForeground(NOTIFICATION_ID, notification);
        serviceRunning = true;

        return START_STICKY;
    }

    /**
     * Provides the interface to communicate with the service from outside of this class.
     *
     * {@inheritDoc}
     */
    @Override
    public IBinder onBind(Intent intent) {
        return binder;
    }

    /**
     * Sends the given IP package into the internet, if allowed.
     *
     * Parses an IP package into an object and checks whether the destination of the given IP package is allowed.
     * If so sends the package with an existing connection or creates a new one to send it out.
     * @param ipPacketBuffer Raw IP package date
     * @param ipPacketLength length of the IP package
     * @throws IllegalRawDataException
     * @throws IOException
     */
    private void handleAppToInet(byte[] ipPacketBuffer, int ipPacketLength) throws IllegalRawDataException, IOException {
        // Filter real from empty Packages
        ByteBuffer tempBuffer = ByteBuffer.allocate(ipPacketLength);
        tempBuffer.put(ipPacketBuffer, 0, ipPacketLength);
        ipPacketBuffer = tempBuffer.array();

        // generate package, throws exception on error and leaves method
        IpPacket pkg = (IpPacket) IpSelector.newPacket(ipPacketBuffer, 0, ipPacketLength);
        Log.i(TAG, "pkg: " + pkg.toString());

        if (hasForbidden(pkg)) {
            redirectForbiddenRequest(pkg);
            return;
        }

        // send the pkg on an already existing connection, if there is any
        sendWithExistingConnection(pkg);

        // if not, make new connection
        boolean isTcp = pkg.getHeader().getProtocol() == IpNumber.TCP;
        boolean tcpPortExists = false;
        if (isTcp) {
            tcpPortExists = mTcpChannels.contains(((TcpPacket) pkg.getPayload()).getHeader().getSrcPort().valueAsInt());
        }

        if (!isTcp || !tcpPortExists) {
            newConnection(pkg);
        }
    }

    /**
     * Sends the given IP package with the corresponding connection.
     *
     * Returns an ACK package to the userspace if the package as been send successfully. Otherwise a
     * FIN package is returned if the connection has already been closed.
     * @param pkg The package to send
     */
    private void sendWithExistingConnection(IpPacket pkg) {
        Set<SelectionKey> allKeys = mSelector.selectedKeys();
        Log.i(TAG, "openconnections: " + allKeys.size());

        // find established connection
        for (SelectionKey selectionKey : allKeys) {
            //Log.i(TAG, "selectionkey available: " + selectionKey.channel());
            Connection connection = (Connection) selectionKey.attachment();
            if (pkg.getHeader().getProtocol() == IpNumber.TCP) {
                TcpPacket tcpPacket = (TcpPacket) pkg.getPayload();

                if (connection.getSrcPort() == tcpPacket.getHeader().getSrcPort().valueAsInt()) {
                    Log.i(TAG, "pkg existing conn");
                    if (connection instanceof UdpConnection) {
                        Log.i(TAG, "sendWithExistingConnection: got UDP instead of TCP");
                        continue;
                    }
                    TcpConnection tcpConnection = (TcpConnection) connection;

                    // send out
                    if (tcpPacket.getPayload() != null) {
                        SocketChannel socketChannel = (SocketChannel) selectionKey.channel();
                        try {
                            if (socketChannel.finishConnect()) {
                                tcpConnection.send(pkg.getPayload().getPayload().length());

                                //if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                                //    Log.i(TAG, "payloadlength: " + pkg.getPayload().getPayload().length());
                                //    Log.i(TAG, "alreadysend: " + Integer.toUnsignedString(tcpConnection.getSeqNr()));
                                //}

                                IpV4Packet ackPkg = PkgUtils.createAckPkg(pkg, tcpConnection);

                                try {
                                    //Log.i(TAG, "ackPkg: " + ackPkg);
                                    //Log.i(TAG, "ackPkg tcpconnection: " + tcpConnection);
                                    out.write(ackPkg.getRawData());
                                } catch (IOException e) {
                                    e.printStackTrace();
                                }

                                socketChannel.write(ByteBuffer.wrap(pkg.getPayload().getPayload().getRawData()));
                            }
                        } catch (ClosedChannelException e) {
                            e.printStackTrace();
                            tcpConnection.terminate();
                        } catch (IOException e) {
                            Log.e(TAG, "couldn't send to existing conn: ");
                            e.printStackTrace();
                        }
                    } else {
                        //Log.i(TAG, "handleAppToInet: no payload, looking for control inormation: " + tcpPacket);
                        if (tcpPacket.getHeader().getFin()) {
                            tcpConnection.terminate();
                        }
                    }
                }
            }
        }
    }

    /**
     * Creates a new connection for the given Package.
     *
     * Also checks for a forbidden DNS requests and gives out a tailored answer if so.
     * @param pkg
     */
    private void newConnection(IpPacket pkg) {
        SelectionKey key;

        try {
            if (pkg.getHeader().getProtocol() == IpNumber.UDP) {
                key = createChannel(pkg);
                DatagramChannel datagramChannel = (DatagramChannel) key.channel();
                UdpConnection udpConnection = (UdpConnection) key.attachment();

                if (udpConnection.dstPort == 53) {
                    boolean isBlocked = false;
                    DnsPacket.DnsHeader dnsHeader = (DnsPacket.DnsHeader) pkg.getPayload().getPayload().getHeader();
                    ArrayList<DnsDomainName> dnsDomainNames = new ArrayList<>();

                    for (DnsQuestion dnsQuestion : dnsHeader.getQuestions()) {
                        Log.i(TAG, "dnsquest: " + dnsQuestion.getQName().toString());
                        dnsDomainNames.add(dnsQuestion.getQName());

                        if (filter.domainBlocked(dnsQuestion.getQName().toString())) {
                            Log.i(TAG, "isblocked");
                            int port = udpConnection.getSrcPort();
                            isBlocked = true;
                            break;
                        }
                    }

                    if (isBlocked) {
                        DnsPacket dns = PkgUtils.createRedirectedDnsAnswer(pkg, dnsDomainNames, currRedirectIp);
                        domainNames.put(currRedirectIp, dnsDomainNames);

                        currRedirectIp = PkgUtils.incrementLastIpOctet(currRedirectIp);
                        Log.i(TAG, "dnspack: " + dns);
                        IpPacket dnsOut = PkgUtils.createDnsPacket(dns, udpConnection);
                        Log.i(TAG, "dnspackout " + dnsOut);
                        out.write(dnsOut.getRawData());
                        return;
                    }
                }

                if (pkg.getPayload().length() > 0) {
                    try {
                        datagramChannel.write(ByteBuffer.wrap(pkg.getPayload().getPayload().getRawData()));
                    } catch (NullPointerException e) {
                        datagramChannel.close();
                        Log.e(TAG, "newConnection: payload is NullPointer");
                        e.printStackTrace();
                    }
                }
            } else if (pkg.getHeader().getProtocol() == IpNumber.TCP) {
                TcpPacket tcpPacket = (TcpPacket) pkg.getPayload();

                if (tcpPacket.getHeader().getSyn()) {
                    key = createChannel(pkg);
                    TcpConnection tcpConnection = (TcpConnection) key.attachment();

                    mTcpChannels.add(tcpConnection.getSrcPort());

                    IpV4Packet synAckPkg = PkgUtils.createSynAckPkg(pkg, tcpConnection);
                    Log.i(TAG, "synack: " + synAckPkg.toString());
                    out.write(synAckPkg.getRawData());
                }
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    /**
     * Checks all connections for termination flag or timeout.
     *
     * Generates FIN or ACK packages for TCP connections as well.
     */
    public void terminateChannels() {
        Set<SelectionKey> allKeys = mSelector.keys();
        Iterator<SelectionKey> keyIterator = allKeys.iterator();
        SelectionKey channelKey;

        while (keyIterator.hasNext()) {
            channelKey = keyIterator.next();
            Connection connection = (Connection) channelKey.attachment();

            // check if connection is marked for disposal
            if (!connection.isTerminated() && !connection.isTimedOut()) {
                continue;
            }

            // check if we are handling a tcpconnection
            if (connection.getTransport() == Connection.Transport.TCP) {
                TcpConnection tcpConnection = (TcpConnection) connection;

                IpV4Packet lastPkg = null;
                if (tcpConnection.isTerminatedByServer()) {
                    lastPkg = PkgUtils.createFinPkg(tcpConnection.getPkg(), tcpConnection);
                } else {
                    tcpConnection.received(1);
                    lastPkg = PkgUtils.createAckPkg(tcpConnection.getPkg(), tcpConnection);
                }

                try {
                    out.write(lastPkg.getRawData());
                } catch (IOException e) {
                    e.printStackTrace();
                }

                mTcpChannels.remove(tcpConnection.getSrcPort());
            }

            try {
                channelKey.channel().close();
            } catch (IOException e) {
                e.printStackTrace();
            }

            channelKey.cancel();
        }
    }

    /**
     * Checks all open connections for new received data and returns it to the userspace.
     * @throws IOException
     */
    private void handleInetToApp() throws IOException {
        byte[] channelPacketBytes = new byte[0];

        mSelector.selectNow(); // wake up, check for available channel, not blocking
        Set<SelectionKey> keySet = mSelector.selectedKeys(); // select available channel

        for (SelectionKey channelKey : keySet) {
            if (channelKey.isReadable()) {
                ByteBuffer channelPacket = ByteBuffer.allocate(65000);

                int read = readDataFromChannel(channelKey, channelPacket);

                if (read <= 0) {
                    //channelKey.channel().close();
                    //channelKey.cancel();
                    Log.i(TAG, "handleInetToApp: nothing received");
                    if (channelKey.attachment() instanceof TcpConnection) {
                        ((TcpConnection) channelKey.attachment()).terminateFromServer();
                    } else if (channelKey.attachment() instanceof UdpConnection) {
                        ((Connection) channelKey.attachment()).terminate();
                    }
                }

                if (read > 0) {
                    IpPacket outPackage = processReadPackage(channelKey, channelPacket, read);
                    try {
                        out.write(outPackage.getRawData());
                    } catch (IOException e) {
                        e.printStackTrace();
                    }
                }
            }
        }
    }

    /**
     * Creates a new communication channel based on the data in the given IP package.
     * @param pkg Provides information for the newly created channel
     * @return
     * @throws IOException
     */
    private SelectionKey createChannel(IpPacket pkg) throws IOException {
        SelectionKey key = null;
        if (pkg.getHeader().getProtocol() == IpNumber.UDP) {
            UdpConnection udpConnection = (UdpConnection) PkgUtils.extractConnectionInformation(pkg);

            DatagramChannel datagramChannel = DatagramChannel.open();
            datagramChannel.configureBlocking(false);
            int interestSet = SelectionKey.OP_READ | SelectionKey.OP_WRITE;
            key = datagramChannel.register(mSelector, interestSet, udpConnection);

            if (!protect(datagramChannel.socket())) {
                Log.e(TAG, "newConnection: Could not protect UDP socket");
            } else {
                datagramChannel.connect(
                        new InetSocketAddress(
                                udpConnection.getDstAddr(),
                                udpConnection.getDstPort())
                );
            }
        } else if (pkg.getHeader().getProtocol() == IpNumber.TCP) {
            TcpPacket tcpPacket = (TcpPacket) pkg.getPayload();
            TcpConnection tcpConnection = (TcpConnection) PkgUtils.extractConnectionInformation(pkg);

            SocketChannel socketChannel = SocketChannel.open();
            socketChannel.configureBlocking(false);
            //socketChannel.setOption(StandardSocketOptions.SO_RCVBUF,1450);

            int interestSet = SelectionKey.OP_READ | SelectionKey.OP_WRITE | SelectionKey.OP_CONNECT;
            key = socketChannel.register(mSelector, interestSet, tcpConnection);

            if (!protect(socketChannel.socket())) {
                Log.e(TAG, "newConnection: Could not protect TCP Socket");
            } else {
                InetSocketAddress remoteSocketAdress = new InetSocketAddress(
                        tcpConnection.getDstAddr(),
                        tcpConnection.getDstPort());

                socketChannel.connect(remoteSocketAdress);
            }
        }

        return key;
    }

    /**
     * Reads data from the given channel in the given buffer.
     * @param channelKey Channel to read the data from
     * @param channelPacket Buffer to write the data into
     * @return
     */
    private int readDataFromChannel(SelectionKey channelKey, ByteBuffer channelPacket) {
        int read = 0;

        if (channelKey.attachment() instanceof TcpConnection) {
            SocketChannel socketChannel = (SocketChannel) channelKey.channel();
            TcpConnection tcpConnection = (TcpConnection) channelKey.attachment();
            if (socketChannel.isConnected()) {
                try {
                    read = socketChannel.read(channelPacket);
                    //Log.i(TAG, "readbytes: " + read);
                } catch (IOException e) {
                    read = 0; // catch IOException to close channel gracefully
                    tcpConnection.terminateFromServer();
                    e.printStackTrace();
                }
            }
        } else if (channelKey.attachment() instanceof UdpConnection) {
            DatagramChannel datagramChannel = (DatagramChannel) channelKey.channel();
            Connection connection = (Connection) channelKey.attachment();
            try {
                read = datagramChannel.read(channelPacket);
                datagramChannel.close();
            } catch (IOException e) {
                read = 0;
                connection.terminate(); // put connection on the closeable list, if we cant close normally
                e.printStackTrace();
            }
        }

        return read;
    }

    /**
     * Creates an IpPacket from the received buffer and updates the metadata information of the affiliated
     * connection.
     * @param channelKey SelectionKey that contains the connection metadata and socket
     * @param channelPacket Read package from the remote connection
     * @param read length of the read package
     * @return The constructed IpPackage of the connection
     */
    @Nullable
    private IpPacket processReadPackage(SelectionKey channelKey, ByteBuffer channelPacket, int read) {
        Connection connection = (Connection) channelKey.attachment();
        IpPacket outPackage = null;
        if (connection.transport == Connection.Transport.TCP) {
            SocketChannel socketChannel = (SocketChannel) channelKey.channel();
            TcpConnection tcpConnection = (TcpConnection) channelKey.attachment();

            // Filter real from empty Packages
            ByteBuffer tempBuffer = ByteBuffer.allocate(read);
            tempBuffer.put(channelPacket.array(), 0, read);
            channelPacket = tempBuffer;

            // write back Answer from server
            outPackage = PkgUtils.createTcpPackage(channelPacket, tcpConnection);

            // update ack number after package creation
            tcpConnection.received(read);

            Log.i(TAG, "tcpconnection: " + tcpConnection);
            Log.i(TAG, "outpkg: " + outPackage.toString());

        } else if (connection.transport == Connection.Transport.UDP) {
            DatagramChannel datagramChannel = (DatagramChannel) channelKey.channel();
            UdpConnection udpConnection = (UdpConnection)  channelKey.attachment();
            outPackage = PkgUtils.createUdpPackage(channelPacket, udpConnection);
            Log.i(TAG, "outpkg: " + outPackage.toString());

            // DNS Handling
            if (udpConnection.getDstPort() == 53) {
                IpPacket dnsOutPackage = PkgUtils.createDnsPacket(channelPacket, udpConnection);
                Log.i(TAG, "dnsout: " + dnsOutPackage.toString());
            }
        }
        return outPackage;
    }

    /**
     * Outputs a block message to the userspace
     *
     * Applies only to tcp connections and returns a HTML page with a blockmessage to the user. Applications
     * that communicate with a different protocol will get garbage data which more or less also stops
     * the communication.
     * @param pkg Package to base the communication information on
     * @throws IOException
     */
    private void redirectForbiddenRequest(IpPacket pkg) throws IOException {
        Log.w(TAG, "forbidden domain detected");
        if (pkg.getHeader().getProtocol() == IpNumber.UDP) {
            Log.i(TAG, "redirectForbiddenRequest: got udp package");
            return;
        }

        TcpConnection tcpConnection = (TcpConnection) PkgUtils.extractConnectionInformation(pkg);
        // account for syn+ack
        tcpConnection.send(1).received(1);

        ByteBuffer channelPacket = ByteBuffer.wrap(PkgUtils.BLOCK_MSG.getBytes(Charset.defaultCharset()));
        IpPacket tcpPacket = PkgUtils.createTcpPackage(channelPacket, tcpConnection);
        Log.i(TAG, "blocked package: " + tcpPacket);
        out.write(tcpPacket.getRawData());
    }

    /**
     * Checks whether the given IP package or its containing protocol has a forbidden target.
     * @param pkg Package to examine
     * @return true in case of a forbidden target, false otherwise
     * @throws UnknownHostException
     */
    private boolean hasForbidden(IpPacket pkg) throws UnknownHostException {
        boolean isForbidden = false;
        boolean isTcp = pkg.getHeader().getProtocol() == IpNumber.TCP;

        if (pkg.getPayload().getPayload() == null) {
            return isForbidden;
        }

        //boolean isRedirectIP = pkg.getHeader().getDstAddr().equals(InetAddress.getByName(PkgUtils.REDIRECT_IP));
        boolean isRedirectIP = PkgUtils.AddressInNetwork(pkg.getHeader().getDstAddr(), currRedirectIp, 24);
        //if (pkg.getHeader().getDstAddr().equals(InetAddress.getByName(REDIRECT_IP))) {
        //} else if (isTcp) {
        if (isTcp) {
            TcpPacket tcpPacket = (TcpPacket) pkg.getPayload();
            if (tcpPacket.getHeader().getDstPort() == TcpPort.HTTP) {
                Log.i(TAG, "httppayload: " + PkgUtils.payloadDecode(pkg));
                try {
                    HttpHeader httpHeader = new HttpHeader(pkg.getPayload().getPayload().getRawData());
                    if (isRedirectIP || filter.domainBlocked(httpHeader.getHost())) {
                        isForbidden = true;
                        String appname = appNameFromPort("tcp", tcpPacket.getHeader().getSrcPort().valueAsInt());
                        //updateNotification( httpHeader.getHost() + " von " + appname);
                        increaseAppCount(appname);
                        Log.i(TAG, "httpheader: " + appname + " -- " + httpHeader.getUrl());
                    }
                } catch (HttpHeaderParseError httpHeaderParseError) {
                    isForbidden = false;
                }
            } else if (tcpPacket.getHeader().getDstPort() == TcpPort.HTTPS) {
                if (filter.domainBlocked(pkg.getHeader().getDstAddr().getHostAddress())) {
                    isForbidden = true;
                    String appname = appNameFromPort("tcp", tcpPacket.getHeader().getSrcPort().valueAsInt());
                    //updateNotification( pkg.getHeader().getDstAddr().getHostAddress() + " von " + appname);
                    increaseAppCount(appname);
                    Log.i(TAG, "ipfilter: " + appname + " -- " + pkg.getHeader().getDstAddr().getHostAddress());
                }
            }
        } else if (isRedirectIP) {
            Log.i(TAG, "hasForbidden: forbidden ip " + pkg.getHeader() + " -- " + currRedirectIp);
            ArrayList<DnsDomainName> domain = domainNames.get(pkg.getHeader().getDstAddr());
            if (domain != null) {
                String appname = appNameFromPort("udp", ((UdpPacket) pkg.getPayload()).getHeader().getSrcPort().valueAsInt());
                //updateNotification( domain.get(0).toString() + " von udp " + appname);
                increaseAppCount(appname);
            }
            isForbidden = true;
        }

        return isForbidden;
    }

    /**
     * Increases the count of blocked requests in the given application and updates the notification
     * @param application Increase the count for this application
     */
    private void increaseAppCount(String application) {
        int count = appBlockCount.containsKey(application) ? appBlockCount.get(application) : 0;
        appBlockCount.put(application, count + 1);

        List<Map.Entry<String, Integer>> entries = new LinkedList<Map.Entry<String, Integer>>(appBlockCount.entrySet());

        // Sort the list
        // Sort method taken from: https://stackoverflow.com/a/51971086/1330168
        Collections.sort(entries, new Comparator<Map.Entry<String, Integer>>() {
            public int compare(Map.Entry<String, Integer> o1,
                               Map.Entry<String, Integer> o2) {
                return (o2.getValue()).compareTo(o1.getValue());
            }
        });

        StringBuilder notificationMessage = new StringBuilder();
        for (int i = 0; i < entries.size(); ++i) {
            notificationMessage.append(entries.get(i).getKey() + ": " + entries.get(i).getValue().toString() + "\n");
        }

        createNotification(notificationMessage.toString().trim());
    }

    /**
     * Creates an empty notification
     * @return The constructed notification
     */
    private Notification createNotification() {
        return createNotification("");
    }

    /**
     * Creates a notification that informs the user with the given text and the information that filtering
     * is in progress
     * @param notificationText Text to show in the notification
     * @return The constructed notification
     */
    @NonNull
    private Notification createNotification(String notificationText) {
        return createNotification(notificationText, "Filterung Aktiv");
    }

    /**
     * Creates a notification with the given text and title
     * @param notificationText Text to show in the notification
     * @param titleText Text to show in the notification title
     * @return The constructed notification
     */
    @NonNull
    private Notification createNotification(String notificationText, String titleText) {
        NotificationCompat.Builder mBuilder = new NotificationCompat.Builder(getApplicationContext(), CHANNEL_ID)
                .setSmallIcon(R.drawable.ic_notification)
                .setContentTitle(titleText)
                .setContentText(notificationText)
                .setStyle(new NotificationCompat.BigTextStyle().bigText(notificationText))
                .setPriority(NotificationCompat.PRIORITY_LOW);

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            CharSequence name = "TIME for kids - Filterdienst";
            String description = "Zeigt gefilterte Zugriffe an";
            int importance = NotificationManager.IMPORTANCE_LOW;
            NotificationChannel channel = new NotificationChannel(CHANNEL_ID, name, importance);
            channel.setDescription(description);
            // Register the channel with the system; you can't change the importance
            // or other notification behaviors after this
            NotificationManager notificationManager = getSystemService(NotificationManager.class);
            notificationManager.createNotificationChannel(channel);
        }
        Notification notification = mBuilder.build();

        NotificationManagerCompat notificationManager = NotificationManagerCompat.from(getApplicationContext());

        // notificationId is a unique int for each notification that you must define
        notificationManager.notify(NOTIFICATION_ID, notification);
        return notification;
    }

    //private void updateNotification(String notificationText) {
    //    Notification notification = createNotification("Zugriff gefiltert: " + notificationText);
    //    NotificationManagerCompat.from(getApplicationContext()).notify(NOTIFICATION_ID, notification);
    //}

    /**
     * Determines the app which is using a given port
     * @param protocol UDP or TCP as protocol
     * @param port used port
     * @return Application name
     */
    private String appNameFromPort(String protocol, int port) {
        String out = "Anwendung unbekannt";
        try {
            boolean firstline = true;
            String line = "";

            File connfile = new File("/proc/net/" + protocol);
            BufferedReader reader = new BufferedReader(new FileReader(connfile));

            while ((line = reader.readLine()) != null) {
                Log.i(TAG, "appNameFromPort: " + line);
                if (firstline) {
                    firstline = false;
                    continue;
                }

                String[] fields = line.split("\\s+|:");
                String localport = fields[4];
                String uid = fields[13];
                if (Integer.parseInt(localport, 16) == port) {
                    PackageManager pm = getApplicationContext().getPackageManager();
                    String[] packages = pm.getPackagesForUid(Integer.parseInt(uid));
                    Log.i(TAG, "appNameFromPort: " + protocol + " -- " + localport + " -- " + port + " -- " + uid + " -- " + Arrays.toString(packages));
                    if (packages != null) {
                        ApplicationInfo appinfo = pm.getApplicationInfo(packages[0], PackageManager.GET_META_DATA);
                        out = (String) pm.getApplicationLabel(appinfo);
                    }
                }
            }
        } catch (FileNotFoundException e) {
            e.printStackTrace();
        } catch (IOException e) {
            e.printStackTrace();
        } catch (NullPointerException e) {
            e.printStackTrace();
        } catch (PackageManager.NameNotFoundException e) {
            e.printStackTrace();
        }

        return out;
    }

    @Override
    public void onDestroy() {
        Log.i(TAG, "onDestroy: will stop");
        stopVpn();
        super.onDestroy();
    }

    @Override
    public void onRevoke() {
        Log.i(TAG, "onRevoke: rights revoked");
        stopVpn();
        super.onRevoke();
    }

    /**
     * Stops and cleanly shuts down the VPN connection, so that in can be started later without a problem
     */
    public void stopVpn() {
        if (mThread != null) {
            mThread.interrupt();
        }

        //if (mInterface != null) {
        //    try {
        //        // close interface to make it openable later
        //        mInterface.close();
        //    } catch (IOException e) {
        //        e.printStackTrace();
        //    }
        //    mInterface = null;
        //}

        serviceRunning = false;
        createNotification("", "Filterung Inaktiv");
        stopSelf();
        //super.onDestroy();
    }

    /**
     * Changes the filter status of the given category
     * @param categoryName Change status of this category
     * @param checkBoxstate true if the category shall be filtered, false otherwise
     */
    public void changeCategory(String categoryName, boolean checkBoxstate) {
        Log.i(TAG, "changeCategory: changed");
        filter.changeCategory(categoryName, checkBoxstate);
    }

    /**
     * Provides an outside communication interface with the Service
     *
     * {@inheritDoc}
     */
    public class UserVpnBinder extends Binder {
        UserVpnService getService() {
            return UserVpnService.this;
        }


        /**
         * Intercept remote method calls and check for "onRevoke" code which
         * is represented by IBinder.LAST_CALL_TRANSACTION. If onRevoke message
         * was received, call onRevoke() otherwise delegate to super implementation.
         */
        // onTransact and description taken from: https://stackoverflow.com/a/15731435/1330168
        @Override
        public boolean onTransact(int code, Parcel data, Parcel reply, int flags) throws RemoteException {
            // see Implementation of android.net.VpnService.Callback.onTransact()
            if ( code == IBinder.LAST_CALL_TRANSACTION ) {
                onRevoke();
                return true;
            }
            return super.onTransact( code, data, reply, flags );
        }
    }

}
