package com.example.android.vpntest;

import android.support.annotation.NonNull;

import org.pcap4j.packet.IpPacket;

import java.net.Inet4Address;
import java.net.InetAddress;
import java.sql.Timestamp;
import java.time.Instant;

/**
 * Holds the metadata information of a TCP connection.
 * @author Steven Schalhorn
 * @version 1.0
 */
public class TcpConnection extends Connection {

    private int seqNr;
    private int ackNr;

    private boolean initiated;
    private boolean terminatedByServer;

    TcpConnection(int dstPort, int srcPort, InetAddress srcAddr, InetAddress dstAddr, int seqNr, int ackNr, IpPacket pkg) {
        super(dstPort, srcPort, dstAddr, srcAddr, Transport.TCP, pkg);
        this.seqNr = seqNr;
        this.ackNr = ackNr;
        this.terminatedByServer = false;
    }

    @NonNull
    @Override
    public String toString() {
        return "TcpConnection{" +
                "dstPort=" + dstPort +
                ", srcPort=" + srcPort +
                ", dstAddr=" + dstAddr +
                ", seqNr=" + seqNr +
                ", ackNr=" + ackNr +
                ", initiated=" + initiated +
                '}';
    }

    /**
     * Signals whether the connection has been successfully initiated.
     * @return True if the connection has been successfully initiat, false otherwise.
     */
    boolean isInitiated() {
        return initiated;
    }

    /**
     * Marks the connection as successfully initiated.
     */
    void initiated() {
        this.initiated = true;
    }

    /**
     * Marks the connection as terminated by the server. Import for creation of connection closing packages.
     */
    void terminateFromServer() {
        this.terminate();
        this.terminatedByServer = true;
    }

    /**
     * Signals whether the connection has been terminated by the server.
     * @return
     */
    public boolean isTerminatedByServer() {
        return terminatedByServer;
    }

    /**
     * Keeps track of the bytes already sent. Needed for the calculation of the ACK und SEQ numbers.
     * @param count Bytes sent to the remote server
     * @return reference to the object for chaining
     */
    TcpConnection send(int count) {
        this.seqNr += count;
        return this;
    }

    /**
     * Keeps track of the bytes already received. Needed for the calculation of the ACK und SEQ numbers.
     * @param count Bytes received from the remote server
     * @return reference to the object for chaining
     */
    TcpConnection received(int count) {
        this.ackNr += count;
        return this;
    }

    /**
     * Directly set the SEQ number and therefore the number of bytes already send.
     * @param seqNr Number of bytes already send.
     */
    public void setSeqNr(int seqNr) {
        this.seqNr = seqNr;
    }

    /**
     * Directly set the ACK number and therefore the number of bytes already received.
     * @param ackNr Number of bytes already received
     */
    public void setAckNr(int ackNr) {
        this.ackNr = ackNr;
    }

    /**
     * Get the SEQ number and therefor the number of bytes already send.
     * @return
     */
    int getSeqNr() {
        return seqNr;
    }

    /**
     * Get the ACK number and therefore the number of bytes already received.
     * @return
     */
    int getAckNr() {
        return ackNr;
    }
}
