package com.example.android.vpntest;

import org.pcap4j.packet.IpPacket;

import java.net.InetAddress;
import java.sql.Timestamp;

/**
 * Base class for all transport protocols inside an IP package. Currently the base class for TCP and
 * UDP connections. Provides metadata information of the connection to correctly handle the connection.
 * @author Steven Schalhorn
 * @version 1.0
 */
class Connection {
    int dstPort;
    int srcPort;
    InetAddress dstAddr;
    InetAddress srcAddr;
    Transport transport;
    IpPacket pkg;
    private boolean terminated;
    private Timestamp lastAccess;
    private int MAX_AGE = 2 * 60 * 1000; // 2 Minutes in Milliseconds

    Connection(int dstPort, int srcPort, InetAddress dstAddr, InetAddress srcAddr, Transport transport, IpPacket pkg) {
        this.dstPort = dstPort;
        this.srcPort = srcPort;
        this.dstAddr = dstAddr;
        this.srcAddr = srcAddr;
        this.transport = transport;
        this.pkg = pkg;
        this.terminated = false;
        this.lastAccess = new Timestamp(System.currentTimeMillis());
    }

    /**
     * Used to determine whether a connection has timed out.
     * @return True if the connection has timed out, false otherwise
     */
    boolean isTimedOut() {
        Timestamp now = new Timestamp(System.currentTimeMillis());
        return now.getTime() - lastAccess.getTime() > MAX_AGE;
    }

    /**
     * Used to determine whether a connection has been terminated
     * @return True if the connection has been terminated, false otherwise
     */
    boolean isTerminated() {
        return terminated;
    }

    /**
     * Mark the connection as terminated.
     */
    void terminate() {
        this.terminated = true;
    }

    /**
     * Sets the timestamp of the last interaction to now.
     */
    void updateTimestamp() {
        lastAccess = new Timestamp(System.currentTimeMillis());
    }

    /**
     * Provides the initial package of the connection. Sometimes needed as base to create new IP packages.
     * @return Initial IP package of the connection
     */
    IpPacket getPkg() {
        return pkg;
    }

    /**
     * The destination port of the connection.
     * @return Port number
     */
    int getDstPort() {
        return dstPort;
    }

    /**
     * The source port of the connection.
     * @return Port number
     */
    int getSrcPort() {
        return srcPort;
    }

    /**
     * The source address of the connection.
     * @return Source address
     */
    InetAddress getSrcAddr() {
        return srcAddr;
    }

    /**
     * The destination address of the connection.
     * @return Destination address
     */
    InetAddress getDstAddr() {
        return dstAddr;
    }

    /**
     * Used to determine the type of the connection.
     * @return Connection type
     */
    Transport getTransport() {
        return transport;
    }

    /**
     * Enumeration listing all compatible protocol types.
     */
    public enum Transport {
        TCP,
        UDP
    }
}
