package com.example.android.vpntest;


import android.support.annotation.NonNull;
import android.util.Log;

import org.pcap4j.packet.DnsDomainName;
import org.pcap4j.packet.DnsPacket;
import org.pcap4j.packet.DnsRDataA;
import org.pcap4j.packet.DnsResourceRecord;
import org.pcap4j.packet.IllegalRawDataException;
import org.pcap4j.packet.IpPacket;
import org.pcap4j.packet.IpV4Packet;
import org.pcap4j.packet.TcpPacket;
import org.pcap4j.packet.UdpPacket;
import org.pcap4j.packet.UnknownPacket;
import org.pcap4j.packet.namednumber.DnsClass;
import org.pcap4j.packet.namednumber.DnsResourceRecordType;
import org.pcap4j.packet.namednumber.IpNumber;
import org.pcap4j.packet.namednumber.TcpPort;
import org.pcap4j.packet.namednumber.UdpPort;

import java.net.Inet4Address;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.nio.ByteBuffer;
import java.util.ArrayList;

/**
 * Provides and bundles helper functions for the VPN service.
 * @author Steven Schalhorn
 * @version 1.0
 */
final class PkgUtils {
    /**
     * Start ip of the redirect subnet. Taken from TEST-NET-2
     */
    static final String REDIRECT_IP = "198.51.100.1";
    private static final String TAG = "VPN";

    /**
     * Block page string which is send to the user
     */
    public static final String BLOCK_MSG = "HTTP/1.0 200 OK\r\n" +
            "Server: SimpleHTTP/0.6 Python/3.6.7\r\n" +
            "Date: Sun, 24 Feb 2019 12:12:21 GMT\r\n" +
            "Cache-Control: no-cache\r\n" +
            "Content-type: text/html; charset=utf-8\r\n" +
            "Content-Length: 4056\r\n" +
            "\r\n" +
            "<center><img src='data:image/png;base64,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'><br>" +
            "<h1>Zugriff Verboten</h1></center>";

    /**
     * Make utilityclass not instantiable.
     */
    private PkgUtils() {
    }

    /**
     * Creates a SYN-ACK packe for a freshly established TCP connection.
     * @param pkg ACK package that got send out from the application.
     * @param tcpConnection Connection metadata which have to be updated
     * @return Created SYN-ACK package
     */
    @NonNull
    static IpV4Packet createSynAckPkg(IpPacket pkg, TcpConnection tcpConnection) {
        tcpConnection.send(1); // Syn+Ack Response
        TcpPacket tcpPacket = (TcpPacket) pkg.getPayload();
        TcpPacket.Builder tcpBuilder = new TcpPacket.Builder()
                .srcPort(tcpPacket.getHeader().getDstPort())
                .dstPort(tcpPacket.getHeader().getSrcPort())
                .srcAddr((Inet4Address) pkg.getHeader().getDstAddr())
                .dstAddr((Inet4Address) pkg.getHeader().getSrcAddr())
                .acknowledgmentNumber(tcpConnection.getSeqNr())
                .sequenceNumber(tcpConnection.getAckNr())
                .window((short) 1452)
                .ack(true)
                .syn(true)
                .correctChecksumAtBuild(true)
                .paddingAtBuild(true)
                .correctLengthAtBuild(true);

        tcpConnection.received(1); // ack from us

        return new IpV4Packet.Builder((IpV4Packet) pkg)
                .srcAddr((Inet4Address) pkg.getHeader().getDstAddr())
                .dstAddr((Inet4Address) pkg.getHeader().getSrcAddr())
                .correctChecksumAtBuild(true)
                .correctLengthAtBuild(true)
                .payloadBuilder(tcpBuilder)
                .build();
    }

    /**
     * Reads the connecton information from a IP package and creates the metadata out of it.
     * @param pkg Read information from this package
     * @return Metadata of the connection
     */
    static Connection extractConnectionInformation(IpPacket pkg) {
        Connection connection = null;
        if (pkg.getHeader().getProtocol() == IpNumber.UDP) {
            UdpPacket udpPacket = (UdpPacket) pkg.getPayload();
            connection = new UdpConnection(
                    udpPacket.getHeader().getDstPort().valueAsInt(),
                    udpPacket.getHeader().getSrcPort().valueAsInt(),
                    pkg.getHeader().getDstAddr(),
                    pkg.getHeader().getSrcAddr(),
                    Connection.Transport.UDP, pkg);
        } else if (pkg.getHeader().getProtocol() == IpNumber.TCP) {
            TcpPacket tcpPacket = (TcpPacket) pkg.getPayload();
            connection = new TcpConnection(
                    tcpPacket.getHeader().getDstPort().valueAsInt(),
                    tcpPacket.getHeader().getSrcPort().valueAsInt(),
                    pkg.getHeader().getSrcAddr(),
                    pkg.getHeader().getDstAddr(),
                    tcpPacket.getHeader().getSequenceNumber(),
                    0,
                    pkg);
        }

        return connection;
    }

    /**
     * Creates a FIN package for a closed TCP connection.
     * @param pkg Base the FIN package off of this package
     * @param tcpConnection Base the package metadata off of this connection metadata
     * @return Created FIN package
     */
    @NonNull
    static IpV4Packet createFinPkg(IpPacket pkg, TcpConnection tcpConnection) {
        TcpPacket tcpPacket = (TcpPacket) pkg.getPayload();
        TcpPacket.Builder tcpBuilder = new TcpPacket.Builder()
                .srcPort(tcpPacket.getHeader().getDstPort())
                .dstPort(tcpPacket.getHeader().getSrcPort())
                .srcAddr((Inet4Address) pkg.getHeader().getDstAddr())
                .dstAddr((Inet4Address) pkg.getHeader().getSrcAddr())
                .acknowledgmentNumber(tcpConnection.getSeqNr())
                .sequenceNumber(tcpConnection.getAckNr())
                .window((short) 1452)
                .ack(true)
                .fin(true)
                .syn(false)
                .correctChecksumAtBuild(true)
                .paddingAtBuild(true)
                .correctLengthAtBuild(true);
        return new IpV4Packet.Builder((IpV4Packet) pkg)
                .srcAddr((Inet4Address) pkg.getHeader().getDstAddr())
                .dstAddr((Inet4Address) pkg.getHeader().getSrcAddr())
                .correctChecksumAtBuild(true)
                .correctLengthAtBuild(true)
                .payloadBuilder(tcpBuilder)
                .build();
    }

    /**
     * Creates an ACK package for a send package from the application
     * @param pkg Package to generate the ACK for
     * @param tcpConnection Connection metadata which have to be updated
     * @return
     */
    @NonNull
    static IpV4Packet createAckPkg(IpPacket pkg, TcpConnection tcpConnection) {
        tcpConnection.updateTimestamp();
        TcpPacket tcpPacket = (TcpPacket) pkg.getPayload();
        TcpPacket.Builder tcpBuilder = new TcpPacket.Builder()
                .srcPort(tcpPacket.getHeader().getDstPort())
                .dstPort(tcpPacket.getHeader().getSrcPort())
                .srcAddr((Inet4Address) pkg.getHeader().getDstAddr())
                .dstAddr((Inet4Address) pkg.getHeader().getSrcAddr())
                .acknowledgmentNumber(tcpConnection.getSeqNr())
                .sequenceNumber(tcpConnection.getAckNr())
                .window((short) 1452)
                .ack(true)
                .syn(false)
                .correctChecksumAtBuild(true)
                .paddingAtBuild(true)
                .correctLengthAtBuild(true);
        return new IpV4Packet.Builder((IpV4Packet) pkg)
                .srcAddr((Inet4Address) pkg.getHeader().getDstAddr())
                .dstAddr((Inet4Address) pkg.getHeader().getSrcAddr())
                .correctChecksumAtBuild(true)
                .correctLengthAtBuild(true)
                .payloadBuilder(tcpBuilder)
                .build();
    }

    /**
     * Increments the last byte of an IP address. Overflows but skips broadcast and net address.
     * @param ip Increments this IP
     * @return incremented IP
     * @throws UnknownHostException
     */
    static InetAddress incrementLastIpOctet(InetAddress ip) throws UnknownHostException {
        byte[] b = ip.getAddress();

        do {
            ++b[3];
        } while (b[3] == (byte)0 || b[3] == (byte)255);

        return InetAddress.getByAddress(b);
    }

    /**
     * Checks whether an IP address is inside an network range
     * @param addr Check if this IP is inside a Network
     * @param network IP address of the network
     * @param cidr CIDR of the network
     * @return True if the address is in the network, false otherwise
     */
    static boolean AddressInNetwork(InetAddress addr, InetAddress network, int cidr) {
        int iaddr = ByteBuffer.wrap(addr.getAddress()).getInt();
        int inetwork = ByteBuffer.wrap(network.getAddress()).getInt();
        int mask = -1 << 32 - cidr; // remove from IP max width to get the bytes to shift

        return (iaddr & mask) == (inetwork & mask);
    }

    /**
     * Creates a DNS answer with the given redirect IP for the given domain name
     * @param pkg Package containing the DNS request
     * @param dnsDomainNames Domain name to get the answer for
     * @param redirectIp Use this IP in the answer
     * @return DNS package with the created answer
     * @throws UnknownHostException
     */
    @NonNull
    static DnsPacket createRedirectedDnsAnswer(IpPacket pkg, ArrayList<DnsDomainName> dnsDomainNames, InetAddress redirectIp) throws UnknownHostException {
        DnsRDataA rData = new DnsRDataA.Builder()
                .address((Inet4Address) redirectIp)
                .addressPlainText(false)
                .build();

        ArrayList<DnsResourceRecord> dnsAnswers = new ArrayList<>();
        DnsResourceRecord dnsAnswer = new DnsResourceRecord.Builder()
                .name(dnsDomainNames.get(0))
                .dataType(DnsResourceRecordType.A)
                .dataClass(DnsClass.IN)
                .ttl(299)
                .rData(rData)
                .correctLengthAtBuild(true)
                .build();
        dnsAnswers.add(dnsAnswer);

        return ((DnsPacket) pkg.getPayload().getPayload()).getBuilder()
                .answers(dnsAnswers)
                .response(true)
                .recursionAvailable(false)
                .anCount((short) 1)
                .build();
    }

    /**
     * Creates a complete IP package from a DNS package
     * @param dns Payload for the IP package
     * @param udpConnection Metadata information for the IP package
     * @return IP package with the dns package as payload
     */
    @NonNull
    static IpPacket createDnsPacket(DnsPacket dns, UdpConnection udpConnection) {
        IpPacket dnsOutPackage = null;
        Log.i(TAG, "dns: " + dns);
        UdpPacket.Builder dnsBuilder = null;
        dnsBuilder = new UdpPacket.Builder()
                .srcPort(UdpPort.getInstance((short) udpConnection.getDstPort()))
                .dstPort(UdpPort.getInstance((short) udpConnection.getSrcPort()))
                .srcAddr(udpConnection.getDstAddr())
                .dstAddr(udpConnection.getSrcAddr())
                .correctChecksumAtBuild(true)
                .correctLengthAtBuild(true)
                .payloadBuilder(
                        dns.getBuilder()

                );
        dnsOutPackage = new IpV4Packet.Builder((IpV4Packet) udpConnection.getPkg())
                .srcAddr((Inet4Address) udpConnection.getDstAddr())
                .dstAddr((Inet4Address) udpConnection.getSrcAddr())
                .correctChecksumAtBuild(true)
                .correctLengthAtBuild(true)
                .payloadBuilder(dnsBuilder)
                .build();
        return dnsOutPackage;
    }

    /**
     * Creates a complete IP package from the byte representation of a DNS package
     * @param channelPacket Payload for the IP package
     * @param udpConnection Metadata information for the IP package
     * @return IP package with the dns package as payload
     */
    @NonNull
    static IpPacket createDnsPacket(ByteBuffer channelPacket, UdpConnection udpConnection) {
        IpPacket dnsOutPackage = null;
        DnsPacket dns = null;
        try {
            dns = DnsPacket.newPacket(channelPacket.array(), 0, channelPacket.array().length);
        } catch (IllegalRawDataException e) {
            e.printStackTrace();
        }
        return createDnsPacket(dns, udpConnection);
    }

    /**
     * Creates a complete IP package from the byte representation of an UDP package
     * @param channelPacket Payload for the IP package
     * @param udpConnection Metadata information for the IP package
     * @return IP package with the dns package as payload
     */
    @NonNull
    static IpPacket createUdpPackage(ByteBuffer channelPacket, UdpConnection udpConnection) {
        UdpPacket.Builder udpPayloadBuilder = new UdpPacket.Builder()
                .srcPort(UdpPort.getInstance((short) udpConnection.getDstPort()))
                .dstPort(UdpPort.getInstance((short) udpConnection.getSrcPort()))
                .srcAddr(udpConnection.getDstAddr())
                .dstAddr(udpConnection.getSrcAddr())
                .correctChecksumAtBuild(true)
                .correctLengthAtBuild(true)
                .payloadBuilder(
                        new UnknownPacket.Builder().rawData(channelPacket.array())
                );

        IpPacket outPackage = null;

        outPackage = new IpV4Packet.Builder((IpV4Packet) udpConnection.getPkg())
                .srcAddr((Inet4Address) udpConnection.getDstAddr())
                .dstAddr((Inet4Address) udpConnection.getSrcAddr())
                .correctChecksumAtBuild(true)
                .correctLengthAtBuild(true)
                .payloadBuilder(udpPayloadBuilder)
                .build();

        udpConnection.updateTimestamp();
        return outPackage;
    }

    /**
     * Converts the byte representation of the payload of an IP package to human readable text
     * @param pkg Extract and convert the payload of this package
     * @return Human readable content of the given IP package
     */
    static String payloadDecode(IpPacket pkg) {
        return new String(pkg.getPayload().getPayload().getRawData());
    }

    /**
     * Creates a complete IP package from the byte representation of an TCP package
     * @param channelPacket Payload for the IP package
     * @param tcpConnection Metadata information for the IP package
     * @return IP package with the dns package as payload
     */
    @NonNull
    static IpPacket createTcpPackage(ByteBuffer channelPacket, TcpConnection tcpConnection) {
        tcpConnection.updateTimestamp();
        TcpPacket.Builder tcpBuilder = new TcpPacket.Builder()
                .srcPort(TcpPort.getInstance((short)tcpConnection.getDstPort()))
                .dstPort(TcpPort.getInstance((short)tcpConnection.getSrcPort()))
                .srcAddr(tcpConnection.getDstAddr())
                .dstAddr(tcpConnection.getSrcAddr())
                .acknowledgmentNumber(tcpConnection.getSeqNr())
                .sequenceNumber(tcpConnection.getAckNr())
                .window((short) 1452)
                .ack(true)
                .syn(false)
                .psh(true)
                .correctChecksumAtBuild(true)
                .paddingAtBuild(true)
                .correctLengthAtBuild(true)
                .payloadBuilder(
                        new UnknownPacket.Builder().rawData(channelPacket.array())
                );

        IpPacket outPackage = null;
        outPackage = new IpV4Packet.Builder((IpV4Packet) tcpConnection.getPkg())
                .srcAddr((Inet4Address) tcpConnection.getDstAddr())
                .dstAddr((Inet4Address) tcpConnection.getSrcAddr())
                .correctChecksumAtBuild(true)
                .correctLengthAtBuild(true)
                .payloadBuilder(tcpBuilder)
                .build();

        return outPackage;
    }
}
