package com.example.android.vpntest;

import android.content.Context;
import android.content.SharedPreferences;
import android.util.Log;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.regex.Pattern;

/**
 * Loads and maintains several different lists of blocked domains and IPs in several categories.
 * Provides information whether a resource is blocked or not as well
 * @author Steven Schalhorn
 * @version 1.0
 */
public class ListFilter {
    private static final String TAG = "VPN";
    private static HashSet<String> domains;
    private final ConcurrentHashMap<String, Object> hashDomains;
    private static final Object PRESENT = new Object();
    private static ExecutorService executor = Executors.newFixedThreadPool(1);
    private static Context context;


    /**
     * Constructs the filter and configures it to filter domains in the given categories.
     * @param context Environmental data of the Android application to save the filtered categories
     * @param categories Filter these categories
     */
    public ListFilter(final Context context, ArrayList<Integer> categories) {
        this.context = context;
        hashDomains = new ConcurrentHashMap<String, Object>();
        domains = new HashSet<String>(100000000); // enough to nearly hold the whole internet
        final ArrayList<String> categoryNames = new ArrayList<String>();

        SharedPreferences sharedPref = context.getSharedPreferences("VPN", Context.MODE_PRIVATE);
        String enabledCategories;
        for (Integer category : categories) {
            boolean categoryState = sharedPref.getBoolean(category.toString(), true);
            if (categoryState) {
                String categoryName = context.getResources().getResourceEntryName(category); // get id name of resource intger
                categoryNames.add(categoryName);
            }
        }

        modifyFilterdomains(categoryNames, true);
    }

    /**
     * Sets the filter to filter or not filter the given categories.
     * @param categoryNames Apply settings to domains or IPs from these categories.
     * @param add expand list of filtered categries if true, remove from the list otherwise
     */
    private void modifyFilterdomains(ArrayList<String> categoryNames, final boolean add) {
        try {
            final String[] assets = context.getAssets().list("");
            for (final String categoryName : categoryNames) {
                for (final String asset : assets) {
                    if (! asset.startsWith(categoryName)) {
                        continue; // move on if file doesn't belong to category
                    }

                    Runnable listRunnable = new Runnable() {
                        @Override
                        public void run() {
                            Log.i(TAG, "ListFilter: " + asset);

                            try {
                                BufferedReader listReader = new BufferedReader(
                                        new InputStreamReader(context.getAssets().open(asset), "ASCII"));

                                while (listReader.ready()) {
                                    String domain = listReader.readLine().trim();
                                    if (domain.startsWith("#") || domain.isEmpty() || domain.length() > 1024) {
                                        continue; // dont process comments and empty lines
                                    }

                                    //try {
                                    //    domain = domain.split("\\s+")[1];
                                    //} catch (ArrayIndexOutOfBoundsException e) {
                                    //    Log.e(TAG, "ListFilter: ", e);
                                    //    Log.e(TAG, "ListFilter: " + domain);
                                    //    continue;
                                    //}
                                    if (add) {
                                        hashDomains.put(domain, PRESENT);
                                    } else {
                                        hashDomains.remove(domain);
                                    }
                                }
                                System.gc();
                            } catch (IOException e) {
                                e.printStackTrace();
                            }
                        }
                    };

                    executor.execute(listRunnable);
                }
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    /**
     * Checks whether the given domain is forbidden.
     * @param domain Check this domain
     * @return If forbidden true, otherwise false
     */
    public boolean domainBlocked(String domain) {
        return hashDomains.containsKey(domain);
    }


    /**
     * Outside interface to modify the filter lists
     * @param categoryName Change state for this category
     * @param checkBoxstate If category shall be filtered true, otherwise false
     */
    public void changeCategory(String categoryName, boolean checkBoxstate) {
        ArrayList<String> categoryNames = new ArrayList<String>();
        categoryNames.add(categoryName);
        modifyFilterdomains(categoryNames, checkBoxstate);
    }
}
